/*******************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 *******************************************************************************

   Last Modification:
    @version $Id: SIIIM_HAL.h 36958 2013-07-31 13:39:49Z Sven $

  Changes:

      Version    Date         Author    Description
      ---------------------------------------------------------------------------
      1         11-15.02.2008    sb        created

-------------------------------------------------------------------------------

TODO:

  - Add definitions for cycle supervision

*******************************************************************************/

#ifndef SIIIM_HAL_H_INCLUDED
#define SIIIM_HAL_H_INCLUDED

#include <stdbool.h>

/*---------------------------------------------------------------------------*/
/* Compiler settings */
#if _MSC_VER >= 1000
  #define __SIIIM_HAL_PACKED_PRE
  #define __SIIIM_HAL_PACKED_POST
  #define __SIIIM_HAL_PACKED_POST_ALIGN2
  #pragma once
  #pragma pack(1)            /* Always align structures to byte boundery */
  #ifndef STRICT             /* Check Typdefinition */
    #define STRICT
  #endif
#endif /* _MSC_VER >= 1000 */

/* support for GNU compiler */
#ifdef __GNUC__
  #define __SIIIM_HAL_PACKED_PRE
  #define __SIIIM_HAL_PACKED_POST    __attribute__((packed))
  #define __SIIIM_HAL_PACKED_POST_ALIGN2 __attribute__((packed,aligned(2)))
#endif

/* support for REALVIEW ARM compiler */
#if defined (__ADS__) || defined (__REALVIEW__)
  #define __SIIIM_HAL_PACKED_PRE   __packed
  #define __SIIIM_HAL_PACKED_POST
  #define __SIIIM_HAL_PACKED_POST_ALIGN2
#endif

/******************************************************************************
 * SIIIM LLD handle definition
 */
typedef struct SIIIM_HANDLE_Ttag* SIIIM_HANDLE;

/******************************************************************************
 * Error codes
 */
typedef enum SIIIM_RESULTtag
{
  SIIIM_RESULT_SUCCESS,
  SIIIM_RESULT_FAILED_TO_INITIALIZE_OSAL                                  = 0x10000,
  SIIIM_RESULT_TO_BE_IMPLEMENTED                                          = 0x10001,
  SIIIM_RESULT_OUT_OF_MEMORY                                              = 0x10002,
  SIIIM_RESULT_FAILED_TO_ACQUIRE_PHYS_VIRT_MAPPING                        = 0x10003,
  SIIIM_RESULT_FAILED_TO_RESET_XC                                         = 0x10004,
  SIIIM_RESULT_FAILED_TO_ACQUIRE_SYSTIME_NS_IRQ                           = 0x10005,
  SIIIM_RESULT_FAILED_TO_ACQUIRE_COM0_IRQ                                 = 0x10006,
  SIIIM_RESULT_FAILED_TO_ACQUIRE_COM1_IRQ                                 = 0x10007,
  SIIIM_RESULT_DISABLING_PENDING                                          = 0x10008,
  SIIIM_RESULT_INVALID_SERVICE_CHANNEL                                    = 0x10009,
  SIIIM_RESULT_INVALID_DEVICE                                             = 0x1000A,
  SIIIM_RESULT_INVALID_PHASE_TRANSITION                                   = 0x1000B,
  SIIIM_RESULT_CALLBACK_IN_USE                                            = 0x1000C,
  SIIIM_RESULT_NO_CONFIGURATION_DATA_FOR_CP3_4                            = 0x1000D,
  SIIIM_RESULT_INVALID_MDT0_SIZE                                          = 0x1000E,
  SIIIM_RESULT_INVALID_MDT1_SIZE                                          = 0x1000F,
  SIIIM_RESULT_INVALID_MDT2_SIZE                                          = 0x10010,
  SIIIM_RESULT_INVALID_MDT3_SIZE                                          = 0x10011,
  SIIIM_RESULT_INVALID_AT0_SIZE                                           = 0x10012,
  SIIIM_RESULT_INVALID_AT1_SIZE                                           = 0x10013,
  SIIIM_RESULT_INVALID_AT2_SIZE                                           = 0x10014,
  SIIIM_RESULT_INVALID_AT3_SIZE                                           = 0x10015,
  SIIIM_RESULT_INVALID_DEVICE_CONTROL_OFFSET_TEL_TYPE                     = 0x10016,
  SIIIM_RESULT_INVALID_DEVICE_CONTROL_OFFSET_TEL_NO                       = 0x10017,
  SIIIM_RESULT_INVALID_DEVICE_CONTROL_OFFSET_OFS_TOO_LOW                  = 0x10018,
  SIIIM_RESULT_INVALID_DEVICE_CONTROL_OFFSET_OFS_TOO_HIGH                 = 0x10019,
  SIIIM_RESULT_INVALID_DEVICE_CONTROL_OFFSET_OFS_NOT_EVEN                 = 0x1001A,
  SIIIM_RESULT_INVALID_DEVICE_STATUS_OFFSET_TEL_TYPE                      = 0x1001B,
  SIIIM_RESULT_INVALID_DEVICE_STATUS_OFFSET_TEL_NO                        = 0x1001C,
  SIIIM_RESULT_INVALID_DEVICE_STATUS_OFFSET_OFS_TOO_LOW                   = 0x1001D,
  SIIIM_RESULT_INVALID_DEVICE_STATUS_OFFSET_OFS_TOO_HIGH                  = 0x1001E,
  SIIIM_RESULT_INVALID_DEVICE_STATUS_OFFSET_OFS_NOT_EVEN                  = 0x1001F,
  SIIIM_RESULT_INVALID_MDT_SVC_CHANNEL_OFFSET_TEL_TYPE                    = 0x10020,
  SIIIM_RESULT_INVALID_MDT_SVC_CHANNEL_OFFSET_TEL_NO                      = 0x10021,
  SIIIM_RESULT_INVALID_MDT_SVC_CHANNEL_OFFSET_OFS_TOO_LOW                 = 0x10022,
  SIIIM_RESULT_INVALID_MDT_SVC_CHANNEL_OFFSET_OFS_TOO_HIGH                = 0x10023,
  SIIIM_RESULT_INVALID_MDT_SVC_CHANNEL_OFFSET_OFS_NOT_EVEN                = 0x10024,
  SIIIM_RESULT_INVALID_AT_SVC_CHANNEL_OFFSET_TEL_TYPE                     = 0x10025,
  SIIIM_RESULT_INVALID_AT_SVC_CHANNEL_OFFSET_TEL_NO                       = 0x10026,
  SIIIM_RESULT_INVALID_AT_SVC_CHANNEL_OFFSET_OFS_TOO_LOW                  = 0x10027,
  SIIIM_RESULT_INVALID_AT_SVC_CHANNEL_OFFSET_OFS_TOO_HIGH                 = 0x10028,
  SIIIM_RESULT_INVALID_AT_SVC_CHANNEL_OFFSET_OFS_NOT_EVEN                 = 0x10029,
  SIIIM_RESULT_INVALID_MDT_RTDATA_OFFSET_TEL_TYPE                         = 0x1002A,
  SIIIM_RESULT_INVALID_MDT_RTDATA_OFFSET_TEL_NO                           = 0x1002B,
  SIIIM_RESULT_INVALID_MDT_RTDATA_OFFSET_OFS_TOO_LOW                      = 0x1002C,
  SIIIM_RESULT_INVALID_MDT_RTDATA_OFFSET_OFS_TOO_HIGH                     = 0x1002D,
  SIIIM_RESULT_INVALID_MDT_RTDATA_OFFSET_OFS_NOT_EVEN                     = 0x1002E,
  SIIIM_RESULT_INVALID_AT_RTDATA_OFFSET_TEL_TYPE                          = 0x1002F,
  SIIIM_RESULT_INVALID_AT_RTDATA_OFFSET_TEL_NO                            = 0x10030,
  SIIIM_RESULT_INVALID_AT_RTDATA_OFFSET_OFS_TOO_LOW                       = 0x10031,
  SIIIM_RESULT_INVALID_AT_RTDATA_OFFSET_OFS_TOO_HIGH                      = 0x10032,
  SIIIM_RESULT_INVALID_AT_RTDATA_OFFSET_OFS_NOT_EVEN                      = 0x10033,
  SIIIM_RESULT_OVERLAPPING_REGIONS_DETECTED_IN_MDT_FRAMES                 = 0x10034,
  SIIIM_RESULT_OVERLAPPING_REGIONS_DETECTED_IN_AT_FRAMES                  = 0x10035,
  SIIIM_RESULT_INVALID_SLAVE_ADDRESS_IN_CP3_4_CONFIG                      = 0x10036,
  SIIIM_RESULT_CONFIGURE_NOT_ALLOWED_IN_CURRENT_STATE                     = 0x10037,
  SIIIM_RESULT_BUILDING_COPY_ROUTINES_FAILED                              = 0x10038,
  SIIIM_RESULT_INVALID_MDT_CONNCTRL_PROCESS_IMAGE_OFFSET                  = 0x10039,
  SIIIM_RESULT_INVALID_MDT_RTDATA_PROCESS_IMAGE_OFFSET                    = 0x1003A,
  SIIIM_RESULT_INVALID_AT_CONNCTRL_PROCESS_IMAGE_OFFSET                   = 0x1003B,
  SIIIM_RESULT_INVALID_AT_RTDATA_PROCESS_IMAGE_OFFSET                     = 0x1003C,
  SIIIM_RESULT_INVALID_HP_CHANNEL                                         = 0x1003D,
  SIIIM_RESULT_MDT_BUFFER_OUT_OF_SYNC                                     = 0x1003E,
  SIIIM_RESULT_TOPOLOGY_CHANGED_DURING_RING_DELAY_MEASUREMENT             = 0x1003F,
  SIIIM_RESULT_NOT_ALLOWED                                                = 0x10040,
  SIIIM_RESULT_SLAVE_NOT_AVAILABLE                                        = 0x10041,
  SIIIM_RESULT_INVALID_PARAMETER                                          = 0x10042,

  SIIIM_RESULT_TIMING_PARAMETER_NRT_CHANNEL_EXCEEDS_CYCLE_TIME            = 0x10043,
  SIIIM_RESULT_TIMING_PARAMETER_END_OF_MDT_IS_LATER_THAN_START_OF_AT      = 0x10044,
  SIIIM_RESULT_TIMING_PARAMETER_END_OF_MDT_EXCEEDS_CYCLE_TIME             = 0x10045,
  SIIIM_RESULT_TIMING_PARAMETER_START_OF_AT_EXCEEDS_CYCLE_TIME            = 0x10046,
  SIIIM_RESULT_TIMING_PARAMETER_END_OF_AT_EXCEEDS_CYCLE_TIME              = 0x10047,
  SIIIM_RESULT_TIMING_PARAMETER_END_OF_NRT_IS_EARLIER_THAN_START_OF_NRT   = 0x10048,
  SIIIM_RESULT_TIMING_PARAMETER_NRT_CHANNEL_OVERLAPS_MDT_TRANSMISSION     = 0x10049,
  SIIIM_RESULT_TIMING_PARAMETER_NRT_CHANNEL_OVERLAPS_AT_TRANSMISSION      = 0x1004A,
  SIIIM_RESULT_TIMING_PARAMETER_START_OF_AT_IS_EARLIER_THAN_START_OF_MDT  = 0x1004B,
  SIIIM_RESULT_LOCKED_DUE_PREVIOUS_FATAL_ERROR                            = 0x1004C,
  SIIIM_RESULT_INVALID_IFG_TIME                                           = 0x1004D,
  SIIIM_RESULT_TIMING_PARAMETER_END_OF_NRT_EXCEEDS_CYCLE_TIME             = 0x1004E,
  SIIIM_RESULT_EXCEPTION_OCCURED                                          = 0x1004F,    /* only used on special implementations */
  SIIIM_RESULT_CONNECTION_LOST                                            = 0x10050,    /* only used on special implementations */
  SIIIM_RESULT_REQUEST_TIMEOUT                                            = 0x10051,    /* only used on special implementations */
  SIIIM_RESULT_ACCESS_ERROR                                               = 0x10052,
  SIIIM_RESULT_SVC_CONTAINER_BUSY                                         = 0x10053,
  SIIIM_RESULT_MAILBOX_ERROR                                              = 0x10054,
  SIIIM_RESULT_FAILED_TO_ACQUIRE_RECV_TIMER_IRQ                           = 0x10055,
  SIIIM_RESULT_INVALID_SYSTIME_CONTROL_PROCESS_IMAGE_OFFSET               = 0x10056,
  SIIIM_RESULT_INVALID_SYSTIME_STATUS_PROCESS_IMAGE_OFFSET                = 0x10057
} SIIIM_RESULT;


/******************************************************************************
 * RxTimer handling (if not specified to SIIIM_Initialize only compatibility mode to S3M V2.0.X is enabled)
 */
typedef struct
{
  bool (* timer_init)();
  void (* timer_deinit)();
  void (* timer_start)(unsigned long ulDelayTimeTicks); /* must be called everytime */
  void (* timer_stop)();
  void (* timer_irq_ack)();
  unsigned long ulNsToTicksDivider; /* do not handle division within cyclic handling, therefore divider placed here for pre-calculation */
  unsigned long ulIrqSource;
} SIIIM_RECV_TIMER_MODULE_T;

/* common */
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioTimer0; /* not available on rcX when using netX50/100/500 */
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioTimer1;
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioTimer2;
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioTimer3;
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioTimer4;

/* netX50/100/500 */
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_GpioSystimeNs;

/* netX51 */
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_ArmTimer0; /* not available on rcX */
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_ArmTimer1;
extern const SIIIM_RECV_TIMER_MODULE_T g_tSIIIM_RecvTimer_ArmTimerSystimeNs;

/******************************************************************************
 * Setup/Cleanup
 */

typedef struct SIIIM_PARAMETERS_Ttag
{
  unsigned char     abSrcMacAddr[6];
  unsigned short    usSvc_MHS_AHS_TimeoutInCycles;
  unsigned long     ulSvc_BUSY_TimeoutInCycles;
  const SIIIM_RECV_TIMER_MODULE_T* ptRecvTimer; /* (if not specified to SIIIM_Initialize only compatibility mode to S3M V2.0.X is enabled) */
} SIIIM_PARAMETERS_T;

SIIIM_HANDLE
SIIIM_Initialize(
                  const SIIIM_PARAMETERS_T*   ptParameters,
                  SIIIM_RESULT*               ptResult);

void
SIIIM_Deinitialize(
                  SIIIM_HANDLE            hHandle);

#ifndef __REMOVE_LIC__
unsigned long
SIIIM_Check(          SIIIM_HANDLE        hHandle);
#endif

/* special function defined by certain osals */
SIIIM_HANDLE
SIIIM_GetInterface();

/* sets a function for a short sleep when waiting for HAL events */
void
SIIIM_SetWaitFunction(SIIIM_HANDLE        hHandle,
                      void(*              pfnWaitFunc)(void));

void
SIIIM_SetSeqCntInitialValues(
                      SIIIM_HANDLE        hHandle,
                      unsigned short      usSeqCntPrimary,
                      unsigned short      usSeqCntSecondary);

void
SIIIM_GetSeqCntInitialValues(
                      SIIIM_HANDLE        hHandle,
                      unsigned short*     pusSeqCntPrimary,
                      unsigned short*     pusSeqCntSecondary);


typedef enum SIIIM_DST_MAC_MODE_Etag
{
  SIIIM_DST_MAC_MODE_E_UNICAST,
  SIIIM_DST_MAC_MODE_E_MULTICAST,
  SIIIM_DST_MAC_MODE_E_BROADCAST
} SIIIM_DST_MAC_MODE_E;

void
SIIIM_SetDstMacMode(
                  SIIIM_HANDLE            ptHandle,
                  SIIIM_DST_MAC_MODE_E    eDstMacMode);

SIIIM_DST_MAC_MODE_E
SIIIM_GetDstMacMode(
                  SIIIM_HANDLE            ptHandle);

#define SIIIM_HAVE_DST_MAC_MODE_CONTROL   /* define for showing defined functionality */

/******************************************************************************
 * Inter Frame Gap Configuration
 */

SIIIM_RESULT
SIIIM_SetS3IfgSize(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulIfgSize);

SIIIM_RESULT
SIIIM_SetNrtIfgSize(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulIfgSize);

void
SIIIM_GetPresetIfgSizes(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulIfgSizeNs_CP1_2,
                  unsigned long*          pulIfgSizeNs_CP3_4);

void
SIIIM_GetXcIfgSizes(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulS3IfgSizeNs_Xc0,
                  unsigned long*          pulS3IfgSizeNs_Xc1,
                  unsigned long*          pulNrtIfgSizeNs_Xc0,
                  unsigned long*          pulNrtIfgSizeNs_Xc1);

void
SIIIM_GetXcIfgSizesInternal(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulS3IfgSizeNs_Xc0,
                  unsigned long*          pulS3IfgSizeNs_Xc1,
                  unsigned long*          pulNrtIfgSizeNs_Xc0,
                  unsigned long*          pulNrtIfgSizeNs_Xc1);

/******************************************************************************
 * Reconfiguration of initialization parameters
 *
 * following functions are only allowed if the master is in NRT
 */

void
SIIIM_SetBusyTimeoutCycles(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulSvc_BUSY_TimeoutInCycles);

unsigned long
SIIIM_GetBusyTimeoutCycles(
                  SIIIM_HANDLE            ptHandle);

void
SIIIM_SetMhsAhsTimeoutCycles(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulSvc_MhsAhs_TimeoutInCycles);

unsigned long
SIIIM_GetMhsAhsTimeoutCycles(
                  SIIIM_HANDLE            ptHandle);

void
SIIIM_SetDevStatusLostLinkTimeoutCycles(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulDevStatusLostLinkCycles);

unsigned long
SIIIM_GetDevStatusLostLinkTimeoutCycles(
                  SIIIM_HANDLE            ptHandle);

void
SIIIM_SetSvcStatusLostLinkTimeoutCycles(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulSvcStatusLostLinkCycles);

unsigned long
SIIIM_GetSvcStatusLostLinkTimeoutCycles(
                  SIIIM_HANDLE            ptHandle);

void
SIIIM_SetCommunicationVersion(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulSercosVersion,
                  unsigned long           ulSercosVersionModifyMask);

unsigned long
SIIIM_GetCommunicationVersion(
                  SIIIM_HANDLE            hHandle);

#define MSK_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_MDT_AT_COUNT       0x00030000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_MDT_AT_COUNT_2     0x00000000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_MDT_AT_COUNT_4     0x00010000

#define MSK_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_PARA_TX            0x00100000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_PARA_TX_OFF        0x00000000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_CP1_2_PARA_TX_ON         0x00100000

#define MSK_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_FAST_CP_SWITCH           0x00200000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_FAST_CP_SWITCH_OFF       0x00000000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_FAST_CP_SWITCH_ON        0x00200000

#define MSK_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_DISABLE_S3_FW_ON_LB      0x00400000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_DISABLE_S3_FW_ON_LB_OFF  0x00000000
#define VAL_SIIIM_MDT_FRAME_CP0_SERCOS_VERSION_DISABLE_S3_FW_ON_LB_ON   0x00400000

void
SIIIM_SetAdvertisedCP1_2_Parameters(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulATStartTime,
                  unsigned long           ulNrtStartTime,
                  unsigned long           ulNrtEndTime);

void
SIIIM_SetMacAddress(
                  SIIIM_HANDLE            hHandle,
                  const unsigned char*    pbMacAddr);

void
SIIIM_GetMacAddress(
    SIIIM_HANDLE hHandle,
    unsigned char* pbMacAddr);

/******************************************************************************
 * Ring Delay Calculation
 */
void
SIIIM_ResetRingDelayMeasurement(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_GetRingDelayMeasurement(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulDelayPrimaryPort,
                  unsigned long*          pulDelaySecondaryPort,
                  unsigned long*          pulNumSamples);

void
SIIIM_GetRingDelayLastCycle(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulDelayPrimaryPort,
                  unsigned long*          pulDelaySecondaryPort);

/******************************************************************************
 * Error Detection
 */

SIIIM_RESULT
SIIIM_SetupFrameLengthErrorInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnFrameLengthError)(
                                          SIIIM_HANDLE        hHandle,
                                          unsigned char       bPhase,
                                          unsigned char       bMDTAT_Type,
                                          void*               pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteFrameLengthErrorInterrupt(
                  SIIIM_HANDLE            hHandle);

void
SIIIM_SetupNotUpdatedErrorInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnNotUpdatedError)(
                                          SIIIM_HANDLE        hHandle,
                                          unsigned int        uiReason,
                                          void*               pvUserData),
                  void*                   pvUserData);

#define SIIIM_NOT_UPDATED_PRIMARY_PORT_MDT        0
#define SIIIM_NOT_UPDATED_PRIMARY_PORT_AT         1
#define SIIIM_NOT_UPDATED_SECONDARY_PORT_MDT      2
#define SIIIM_NOT_UPDATED_SECONDARY_PORT_AT       3

void
SIIIM_DeleteNotUpdatedErrorInterrupt(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Fatal Error Detection
 */
SIIIM_RESULT
SIIIM_SetupFatalErrorInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void (*                 pfnFatalError)(
                                          SIIIM_HANDLE        hHandle,
                                          unsigned int        uiReason,
                                          void*               pvUserData),
                  void*                   pvUserData);

enum SIIIM_FATAL_ERROR_REASONtag
{
  SIIIM_FATAL_ERROR_UNKNOWN = 0,
  SIIIM_FATAL_ERROR_MDT_HAS_UNEXPECTED_FRAME_BUFFER,
  SIIIM_FATAL_ERROR_AT_IS_MISSING_FRAME_BUFFER,
  SIIIM_FATAL_ERROR_FRAME_OUT_OF_SEQUENCE,
  SIIIM_FATAL_ERROR_DLPDU_TYPE_MISMATCH,
  SIIIM_FATAL_ERROR_PORT_BUFFERS_NOT_SYNCHRONOUS
};

void
SIIIM_DeleteFatalErrorInterrupt(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Cycle Timer
 */

void
SIIIM_StartTimer(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulCyclesToElapse);

void
SIIIM_StopTimer(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetupTimeoutInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnCycleTimeout)(
                                          SIIIM_HANDLE        hHandle,
                                          void*               pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteTimeoutInterrupt(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Phase Control (Low Level Stuff)
 */

SIIIM_RESULT
SIIIM_SetPhase_NRT(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP0(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP1(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP2(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP3(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP4(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP0S(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP1S(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP2S(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP3S(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetPhase_CP4S(
                  SIIIM_HANDLE            hHandle);

unsigned char
SIIIM_GetCurrentTransmittedPhase(
                  SIIIM_HANDLE            hHandle);

#define MSK_CURRENT_TRANSMITTED_PHASE_CPS     0x80

#define MSK_CURRENT_TRANSMITTED_PHASE_CP      0x0F
#define VAL_CURRENT_TRANSMITTED_PHASE_CP0     0x00
#define VAL_CURRENT_TRANSMITTED_PHASE_CP1     0x01
#define VAL_CURRENT_TRANSMITTED_PHASE_CP2     0x02
#define VAL_CURRENT_TRANSMITTED_PHASE_CP3     0x03
#define VAL_CURRENT_TRANSMITTED_PHASE_CP4     0x04

unsigned long
SIIIM_GetCycleCounter(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Timing Configuration
 */

typedef struct SIIIM_TIMING_DATA_Ttag
{
  /* overall cycle time (in ns) */
  unsigned long     ulCycleTimeNs;
  /* AT start time (in ns, seen from cycle beginning) */
  unsigned long     ulATStartTimeNs;
  unsigned long     ulNRTChannelStartTimeNs;    /* t6, (in ns, seen from cycle beginning)  */
  unsigned long     ulNRTChannelEndTimeNs;      /* t7, (in ns, seen from cycle beginning)  */
} SIIIM_TIMING_DATA_T;


SIIIM_RESULT
SIIIM_SetTimingData(
                  SIIIM_HANDLE            hHandle,
                  const SIIIM_TIMING_DATA_T*
                                          ptTimingData);

unsigned long
SIIIM_GetATRxStartTime(
                  SIIIM_HANDLE           hHandle,
                  const SIIIM_TIMING_DATA_T*
                                         ptTimingData);

unsigned long
SIIIM_GetATRxEndTime(
                  SIIIM_HANDLE           hHandle,
                  const SIIIM_TIMING_DATA_T*
                                         ptTimingData);

unsigned long
SIIIM_GetMDTTxTime(
                  SIIIM_HANDLE           hHandle,
                  const SIIIM_TIMING_DATA_T*
                                         ptTimingData);

/******************************************************************************
 * SCP_SysTime support
 *
 * sercos time = local_time + ulSysTimeOffsetNs
 */
void
SIIIM_SetSysTimeOffset(
                  SIIIM_HANDLE            hHandle,
                  unsigned long           ulSysTimeOffsetNs);

unsigned long
SIIIM_GetSysTimeOffset(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Addressing Mode Configuration
 */

#define SIIIM_CP1_2_ADDRESSING_MODE_TOPOLOGY_ADDRESS          0
#define SIIIM_CP1_2_ADDRESSING_MODE_SERCOS_ADDRESS            1

SIIIM_RESULT
SIIIM_SetDeviceAddressingMode(
                  SIIIM_HANDLE            hHandle,
                  unsigned int            uiAddressingMode);

/******************************************************************************
 *  HotPlug Field Access
 */

/* TODO: HotPlug API definition is not yet complete (consider whether HP differs on primary/secondary) */

typedef struct SIIIM_HP_MDT_Ttag
{
  unsigned short            usSercosAddress;
  unsigned short            usHpControl;
  unsigned long             ulHpInfo;
} SIIIM_HP_MDT_T;

#define MSK_SIIIM_HP_MDT_HOT_PLUG_SUPPORTED           0x8000

#define MSK_SIIIM_HP_MDT_HOT_PLUG_ENABLED             0x0200

#define MSK_SIIIM_HP_MDT_TRANSMISSION_VIA             0x0100
#define VAL_SIIIM_HP_MDT_TRANSMISSION_VIA_HP          0x0000
#define VAL_SIIIM_HP_MDT_TRANSMISSION_VIA_HP_AND_SVC  0x0100

#define MSK_SIIIM_HP_MDT_FUNCTION_CODE                0x00FF
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_NO_DATA        0x0000
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_tScyc          0x0001
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_t6             0x0002
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_t7             0x0003
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MTU            0x0004
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_COMM_VERSION   0x0005
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MDT0_LENGTH    0x0010
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MDT1_LENGTH    0x0011
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MDT2_LENGTH    0x0012
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MDT3_LENGTH    0x0013
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_AT0_LENGTH     0x0020
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_AT1_LENGTH     0x0021
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_AT2_LENGTH     0x0022
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_AT3_LENGTH     0x0023
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_MDT_SVC_OFFSET 0x0080
#define VAL_SIIIM_HP_MDT_FUNCTION_CODE_AT_SVC_OFFSET  0x0081

typedef struct SIIIM_HP_AT_Ttag
{
  unsigned short            usSercosAddress;
  unsigned short            usHpStatus;
  unsigned long             ulHpInfo;
} SIIIM_HP_AT_T;

#define MSK_SIIIM_HP_AT_ERROR_IN_HP1                  0x0100

#define MSK_SIIIM_HP_AT_ACK                           0x00FF
#define VAL_SIIIM_HP_AT_ACK_RESERVED                  0x0000
#define VAL_SIIIM_HP_AT_ACK_NO_DATA                   0x0001
#define VAL_SIIIM_HP_AT_ACK_SVC_ACTIVATION            0x0002
#define VAL_SIIIM_HP_AT_ACK_SERCOS_ADDRESS            0x0004
#define VAL_SIIIM_HP_AT_ACK_SLAVE_SCAN                0x0005
#define VAL_SIIIM_HP_AT_ACK_MDT_SVC_OFFSET            0x0080
#define VAL_SIIIM_HP_AT_ACK_AT_SVC_OFFSET             0x0081
#define VAL_SIIIM_HP_AT_ACK_DUPLICATE_SERCOS_ADDRESS  0x00FF


SIIIM_RESULT
SIIIM_WriteHP(
                  SIIIM_HANDLE            hHandle,
                  const SIIIM_HP_MDT_T*   ptHP_MDT,
                  unsigned int            uiSelectChannel);

#define SIIIM_HP_CHANNEL_PRIMARY          0
#define SIIIM_HP_CHANNEL_SECONDARY        1
#define SIIIM_HP_CHANNEL_BOTH             2     /* only usable for SIIIM_WriteHP */
#define SIIIM_HP_CHANNEL_INACTIVE         3     /* only usable for SIIIM_WriteHP */


#define SIIIM_HP_INT_STATUS_CHANGED               0x0001      /* AT HP field changed */
#define SIIIM_HP_INT_STATUS_MINIMUM_TIME_REACHED  0x0002      /* HP was steady for a minimum of 1-2ms */

SIIIM_RESULT
SIIIM_ReadHP(
                  SIIIM_HANDLE            hHandle,
                  unsigned int            uChannel,
                  SIIIM_HP_AT_T*          ptHP_AT,
                  unsigned short*         pusHPIntStatus);

/* HotPlug Changed interrupt initialization/deinitialization */
SIIIM_RESULT
SIIIM_SetupHotPlugChangedInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnHotPlugChanged)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteHotPlugChangedInterrupt(
                  SIIIM_HANDLE            hHandle);

int
SIIIM_GetLinkStatusOnPrimaryPort(SIIIM_HANDLE hHandle);

int
SIIIM_GetLinkStatusOnSecondaryPort(SIIIM_HANDLE hHandle);

/******************************************************************************
 * Process Data access
 */

/* TODO: Process Data API definition is not yet complete */

SIIIM_RESULT
SIIIM_SetupMDTDataAccessFunctions(
                  SIIIM_HANDLE            hHandle,
                  unsigned char* (*       pfnRequestBuffer)(
                                              SIIIM_HANDLE    hHandle,
                                              void*           pvUserData),
                  void (*                 pfnReleaseBuffer)(
                                              SIIIM_HANDLE    hHandle,
                                              unsigned char*  pabBuffer,
                                              void*           pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteMDTDataAccessFunctions(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_SetupATDataAccessFunctions(
                  SIIIM_HANDLE            hHandle,
                  unsigned char* (*       pfnRequestBuffer)(
                                              SIIIM_HANDLE    hHandle,
                                              void*           pvUserData),
                  void (*                 pfnReleaseBuffer)(
                                              SIIIM_HANDLE    hHandle,
                                              unsigned char*  pabBuffer,
                                              unsigned int    fDataValid,
                                              void*           pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteATDataAccessFunctions(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Device Control/Device Status Access
 */

/* Device Control word */
#define MSK_SIIIM_DEVICE_CONTROL_MASTER_VALID                   0x0100

#define MSK_SIIIM_DEVICE_CONTROL_PHYSICAL_RING_CLOSED           0x0800

#define MSK_SIIIM_DEVICE_CONTROL_TOPOLOGY_CONTROL               0x3000
#define VAL_SIIIM_DEVICE_CONTROL_TOPOLOGY_CONTROL_FAST_FORWARD  0x0000
#define VAL_SIIIM_DEVICE_CONTROL_TOPOLOGY_CONTROL_LOOP_ON_P1    0x1000
#define VAL_SIIIM_DEVICE_CONTROL_TOPOLOGY_CONTROL_LOOP_ON_P0    0x2000

#define MSK_SIIIM_DEVICE_CONTROL_TOPOLOGY_CHANGE_REQUEST_TOGGLE 0x4000

#define MSK_SIIIM_DEVICE_CONTROL_IDENT_REQUEST                  0x8000

/* Device Status word */
#define MSK_SIIIM_DEVICE_STATUS_PARAMETERIZATION_LEVELS         0x0010

#define MSK_SIIIM_DEVICE_STATUS_PROC_COMMAND_CHANGE_BIT         0x0020

#define MSK_SIIIM_DEVICE_STATUS_C2D_WARNING_BIT                 0x0040

#define MSK_SIIIM_DEVICE_STATUS_C1D_ERROR_BIT                   0x0080

#define MSK_SIIIM_DEVICE_STATUS_SLAVE_VALID                     0x0100

#define MSK_SIIIM_DEVICE_STATUS_ERROR_OCCURED                   0x0200

#define MSK_SIIIM_DEVICE_STATUS_INACTIVE_PORT_STATE             0x0C00
#define VAL_SIIIM_DEVICE_STATUS_INACTIVE_PORT_STATE_NO_LINK     0x0000
#define VAL_SIIIM_DEVICE_STATUS_INACTIVE_PORT_STATE_HAS_LINK    0x0400
#define VAL_SIIIM_DEVICE_STATUS_INACTIVE_PORT_STATE_P_FRAMES    0x0800
#define VAL_SIIIM_DEVICE_STATUS_INACTIVE_PORT_STATE_S_FRAMES    0x0C00

#define MSK_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG               0x0C00
#define VAL_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_UNSUPPORTED   0x0000
#define VAL_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_SUPPORTED     0x0400
#define VAL_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_P_CH_ERROR    0x0800
#define VAL_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_S_CH_ERROR    0x0C00

#define MSK_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_ACTIVE        0x0800

#define MSK_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS                 0x3000
#define SRT_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS                 12
#define VAL_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS_FAST_FORWARD    0x0000
#define VAL_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS_LOOP_ON_P1      0x1000
#define VAL_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS_LOOP_ON_P0      0x2000
#define VAL_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS_NRT_MODE        0x3000

#define MSK_SIIIM_DEVICE_STATUS_TOPOLOGY_HS                     0x4000 /* linked to MSK_SIIIM_DEVICE_CONTROL_TOPOLOGY_CHANGE_REQUEST_TOGGLE */

#define MSK_SIIIM_DEVICE_STATUS_COMMUNICATION_WARNING           0x8000

static inline
int
SIIIM_IsPSChannelErrorDiagActive(unsigned short usSDev)
{
  return (usSDev & (MSK_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_ACTIVE | MSK_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS)) ==
      (MSK_SIIIM_DEVICE_STATUS_FAST_FORWARD_DIAG_ACTIVE | VAL_SIIIM_DEVICE_STATUS_TOPOLOGY_STATUS_FAST_FORWARD);
}


/* DevStatus int status */
#define MSK_DEV_STATUS_INT_STATUS_CHANGED                       0x0001
#define MSK_DEV_STATUS_INT_LOST_LINK                            0x0002

/* DeviceStatus Changed bitmap */
typedef struct SIIIM_DEVSTATUS_INTERRUPTS_Ttag
{
  unsigned char       abDevStatusChangedBits[64];
} SIIIM_DEVSTATUS_INTERRUPTS_T;

SIIIM_RESULT
SIIIM_ReadDeviceControl(
                SIIIM_HANDLE              hHandle,
                unsigned short            usSlaveAddress,
                unsigned short*           pusDeviceControl);

/* Device Control/Device Status access functions */
SIIIM_RESULT
SIIIM_ModifyDeviceControl(
                SIIIM_HANDLE              hHandle,
                unsigned short            usSlaveAddress,
                unsigned short            usDeviceControlMask,
                unsigned short            usDeviceControlValue,
                unsigned short            usDeviceControlToggleBits);

SIIIM_RESULT
SIIIM_ModifyTwoDeviceControls(
                SIIIM_HANDLE              hHandle,
                unsigned short            usSlaveAddress1,
                unsigned short            usDeviceControlMask1,
                unsigned short            usDeviceControlValue1,
                unsigned short            usDeviceControlToggleBits1,
                unsigned short            usSlaveAddress2,
                unsigned short            usDeviceControlMask2,
                unsigned short            usDeviceControlValue2,
                unsigned short            usDeviceControlToggleBits2);

SIIIM_RESULT
SIIIM_ReadDeviceStatus(
                SIIIM_HANDLE              hHandle,
                unsigned short            usSlaveAddress,
                unsigned short*           pusDeviceStatus,
                unsigned short*           pusIntStatus);

SIIIM_RESULT
SIIIM_PeekDeviceStatus(
                SIIIM_HANDLE              hHandle,
                unsigned short            usSlaveAddress,
                unsigned short*           pusDeviceStatus);

SIIIM_RESULT
SIIIM_PeekDeviceStatuses(
                SIIIM_HANDLE              hHandle,
                unsigned short            usStartSlaveAddress,
                unsigned short            usNumSlaveAddressesToRead,
                unsigned short*           pusReadSlaveAddresses,
                unsigned short*           pausDeviceStatuses);

/* DeviceStatusChanged bitmap access */
const SIIIM_DEVSTATUS_INTERRUPTS_T*
SIIIM_GetDevStatusChanged(
                  SIIIM_HANDLE            hHandle);

/* DevStatus Changed interrupt initialization/deinitialization */
SIIIM_RESULT
SIIIM_SetupDevStatusChangedInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnDevStatusChanged)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteDevStatusChangedInterrupt(
                  SIIIM_HANDLE            hHandle);

void
SIIIM_SetMasterValid(
                  SIIIM_HANDLE            hHandle,
                  unsigned int            fValid);

unsigned int
SIIIM_GetMasterValid(
                  SIIIM_HANDLE            hHandle);

void
SIIIM_EnableFastSwitch(
                  SIIIM_HANDLE            hHandle,
                  unsigned int            fEnable);

/******************************************************************************
 * Topology Detect Access
 */

#define SIIIM_DETECTED_ADDRESS_COUNT    511

typedef enum SIIIM_DETECTED_FRAMES_ON_PORTtag
{
  SIIIM_NO_FRAMES_ON_PORT,
  SIIIM_P_FRAMES_ON_PORT,
  SIIIM_S_FRAMES_ON_PORT
} SIIIM_DETECTED_FRAMES_ON_PORT;

typedef struct SIIIM_DETECTED_ADDRESSES_Ttag
{
  unsigned short      usPrimarySeqCnt;
  unsigned short      ausPrimaryTopologyAddresses[SIIIM_DETECTED_ADDRESS_COUNT];
  unsigned short      usSecondarySeqCnt;
  unsigned short      ausSecondaryTopologyAddresses[SIIIM_DETECTED_ADDRESS_COUNT];
  SIIIM_DETECTED_FRAMES_ON_PORT tFramesOnPrimaryPort;
  SIIIM_DETECTED_FRAMES_ON_PORT tFramesOnSecondaryPort;
} SIIIM_DETECTED_ADDRESSES_T;

#define MSK_SIIIM_DET_ADDR_SERCOS_ADDRESS           0x01FF
#define MSK_SIIIM_DET_ADDR_REQ_FUNCTIONS_SUPPORTED  0x8000

const SIIIM_DETECTED_ADDRESSES_T*
SIIIM_GetDetectedAddresses(
                  SIIIM_HANDLE            hHandle);

typedef struct SIIIM_RX_PORT_STATUS_Ttag
{
  SIIIM_DETECTED_FRAMES_ON_PORT tFramesOnPrimaryPort;
  SIIIM_DETECTED_FRAMES_ON_PORT tFramesOnSecondaryPort;
} SIIIM_RX_PORT_STATUS_T;

void
SIIIM_GetRxPortStatus(SIIIM_HANDLE            hHandle,
                      SIIIM_RX_PORT_STATUS_T* ptRxPortStatus);

/* TopologyDetect Changed interrupt initialization/deinitialization */
SIIIM_RESULT
SIIIM_SetupTopologyDetectChangedInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnTopologyDetectChanged)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteTopologyDetectChangedInterrupt(
                  SIIIM_HANDLE            hHandle);

/* Frame Received Topology Changed interrupt */
SIIIM_RESULT
SIIIM_SetupFrameReceivedTopologyChangedInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnFrameReceivedTopologyChanged)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteFrameReceivedTopologyChangedInterrupt(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Service Channel Access
 */

/* Service Channel Control Word */
#define MSK_SIIIM_SVC_MDT_MHS             0x0001

#define MSK_SIIIM_SVC_MDT_RW              0x0002
#define VAL_SIIIM_SVC_MDT_RW_READ         0x0000
#define VAL_SIIIM_SVC_MDT_RW_WRITE        0x0002

#define MSK_SIIIM_SVC_MDT_LAST_TRANS      0x0004

#define MSK_SIIIM_SVC_MDT_DBE             0x0038
#define SRT_SIIIM_SVC_MDT_DBE             3
#define VAL_SIIIM_SVC_MDT_DBE_CLOSE       0x0000
#define VAL_SIIIM_SVC_MDT_DBE_IDN         0x0008
#define VAL_SIIIM_SVC_MDT_DBE_NAME        0x0010
#define VAL_SIIIM_SVC_MDT_DBE_ATTRIBUTE   0x0018
#define VAL_SIIIM_SVC_MDT_DBE_UNIT        0x0020
#define VAL_SIIIM_SVC_MDT_DBE_MINIMUM     0x0028
#define VAL_SIIIM_SVC_MDT_DBE_MAXIMUM     0x0030
#define VAL_SIIIM_SVC_MDT_DBE_OPDATA      0x0038

/* Service Channel Status Word */
#define MSK_SIIIM_SVC_AT_AHS              0x0001

#define MSK_SIIIM_SVC_AT_BUSY             0x0002

#define MSK_SIIIM_SVC_AT_SVC_ERROR        0x0004

#define MSK_SIIIM_SVC_AT_SVC_PROCESSING   0x0008


/* Service Channel Structure MDT frame */
typedef struct SIIIM_SVC_MDT_Ttag
{
  unsigned short      usSvcCtrl;
  unsigned long       ulSvcInfo;
} SIIIM_SVC_MDT_T;

/* Service Channel Structure AT frame */
typedef struct SIIIM_SVC_AT_Ttag
{
  unsigned short      usSvcStatus;
  unsigned long       ulSvcInfo;
} SIIIM_SVC_AT_T;

/* Service Channel Changed bitmap */
typedef struct SIIIM_SVC_INTERRUPTS_Ttag
{
  unsigned char       abSvChActiveBits[64];
} SIIIM_SVC_INTERRUPTS_T;

#define MSK_SVC_INT_STATUS_SVC_CHANGED        0x01
#define MSK_SVC_INT_STATUS_MHS_AHS_TIMEOUT    0x02
#define MSK_SVC_INT_STATUS_BUSY_TIMEOUT       0x04
#define MSK_SVC_INT_STATUS_SVC_NOT_VALID      0x08
#define MSK_SVC_INT_STATUS_CONTAINER_DONE     0x10
#define MSK_SVC_INT_STATUS_CONTAINER_ABORTED  0x20

/* access functions for Service Channel */
SIIIM_RESULT
SIIIM_ReadSvcChannelData(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo,
                  SIIIM_SVC_AT_T*         ptSvcData,
                  unsigned char*          pbSvcIntStatus);

SIIIM_RESULT
SIIIM_WriteSvcChannelData(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo,
                  const SIIIM_SVC_MDT_T*  ptSvcData);

/* ServiceChannel Changed bitmap access */
const SIIIM_SVC_INTERRUPTS_T*
SIIIM_GetSvcChannelStatus(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_InitSvcChannel(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo);

/* ServiceChannel Changed interrupt initialization/deinitialization */
SIIIM_RESULT
SIIIM_SetupSvcChannelChangedInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnSvcChannelChanged)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteSvcChannelChangedInterrupt(
                  SIIIM_HANDLE            hHandle);

/* ServiceChannel Changed interrupt initialization/deinitialization (CP1 only) */
SIIIM_RESULT
SIIIM_SetupSvcChannelChangedCP1Interrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnSvcChannelCP1Changed)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteSvcChannelChangedCP1Interrupt(
                  SIIIM_HANDLE            hHandle);

/* ServiceChannel Changed bitmap access */
const SIIIM_SVC_INTERRUPTS_T*
SIIIM_GetSvcChannelValid(
                  SIIIM_HANDLE            hHandle);

unsigned short
SIIIM_GetSvcStatusWord(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo);

/* ServiceChannel Changed interrupt valid signalling */
SIIIM_RESULT
SIIIM_SetupSvcChannelValidInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void(*                  pfnSvcChannelValid)(
                                          SIIIM_HANDLE      hHandle,
                                          void*             pvUserData),
                  void*                   pvUserData);

SIIIM_RESULT
SIIIM_DeleteSvcChannelValidInterrupt(
                  SIIIM_HANDLE            hHandle);

/*
 * Interrupt logic on SVC is as follows:
 *
 * a negative edge on BUSY detected since last frame
 * any edge on AHS
 *
 * if BUSY changes back to 0 without reading by AT0-3, the edge on AHS has to be sufficient.
 * The BUSY is changed to 1 in the same cycle as the last change on MHS.
 * If it has a neg edge before next cycle, we will see a 0 but the AHS will differ from MHS.
 */

/* Following function is for debugging API via CP task (no use in any other task allowed) */
SIIIM_RESULT
SIIIM_PeekSvcChannel(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo,
                  SIIIM_SVC_MDT_T*        ptMdtSvc,
                  SIIIM_SVC_AT_T*         ptAtSvc);


/*
 * SVC Container API
 */

typedef __SIIIM_HAL_PACKED_PRE struct SIIIM_SVC_CONTAINER_CONTROL_ELEMENT_Ttag
{
  __SIIIM_HAL_PACKED_PRE union
  {
    unsigned char bControl;
    unsigned char bStatus;
  } __SIIIM_HAL_PACKED_POST tControlStatus;
} __SIIIM_HAL_PACKED_POST SIIIM_SVC_CONTAINER_CONTROL_ELEMENT_T;

typedef __SIIIM_HAL_PACKED_PRE struct SIIIM_SVC_CONTAINER_DATA_ELEMENT_Ttag
{
  unsigned long ulSvcInfo;
} __SIIIM_HAL_PACKED_POST_ALIGN2 SIIIM_SVC_CONTAINER_DATA_ELEMENT_T;

/* tControlStatus.bControl on request */
/* Bits 6 and 7 are reserved */
#define MSK_SIIIM_SVC_CONTAINER_CONTROL_IGN_SPECIFIC_ERR 0x01      /* Ignore up to 2 error codes which is specified within ulSvcInfo [each of both words low and high]) */
#define MSK_SIIIM_SVC_CONTAINER_CONTROL_RW_FLAG          0x02
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_READ             0x00
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_WRITE            0x02

#define MSK_SIIIM_SVC_CONTAINER_CONTROL_LAST             0x04

#define MSK_SIIIM_SVC_CONTAINER_CONTROL_DBE              0x38
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_CLOSE            0x00
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_IDN              0x08
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_NAME             0x10
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_ATTRIBUTE        0x18
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_UNIT             0x20
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_MINIMUM_VALUE    0x28
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_MAXIMUM_VALUE    0x30
#define VAL_SIIIM_SVC_CONTAINER_CONTROL_OP_DATA          0x38


/* tControlStatus.bStatus on response */
/* Bits 0-1 and 4-7 are reserved */
#define MSK_SIIIM_SVC_CONTAINER_STATUS_HAS_AHS_TIMEOUT   0x01
#define MSK_SIIIM_SVC_CONTAINER_STATUS_HAS_BUSY_TIMEOUT  0x02
#define MSK_SIIIM_SVC_CONTAINER_STATUS_HAS_ERROR         0x04
#define MSK_SIIIM_SVC_CONTAINER_STATUS_LOST_LINK         0x08


SIIIM_RESULT
SIIIM_WriteSvcContainer(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo,
                  unsigned int            uiNumElements,
                  SIIIM_SVC_CONTAINER_CONTROL_ELEMENT_T* ptControlElements,
                  SIIIM_SVC_CONTAINER_DATA_ELEMENT_T* ptDataElements);

SIIIM_RESULT
SIIIM_ReadSvcContainer(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo,
                  unsigned int*           puiNumElements,
                  unsigned int*           puiNumProcessedElements,
                  SIIIM_SVC_CONTAINER_CONTROL_ELEMENT_T** pptControlElements,
                  SIIIM_SVC_CONTAINER_DATA_ELEMENT_T** pptDataElements);

unsigned int
SIIIM_GetMaxContainerElements(
                  SIIIM_HANDLE            hHandle);

SIIIM_RESULT
SIIIM_CancelSvcContainer(
                  SIIIM_HANDLE            hHandle,
                  unsigned short          usSvcChNo);

/******************************************************************************
 * CP3/CP4 MDT/AT configuration functions
 */

/* TODO: CP3/CP4 MDT/AT Configuration API definition not yet complete */

/* Device specific field configuration */
typedef struct SIIIM_DEVICE_OFFSETS_Ttag
{
  unsigned short  usSlaveAddress;

  /* Device Control/Device Status configuration */
  unsigned short  usDevControlOffset;   /* coding according to SIII spec */
  unsigned short  usDevStatusOffset;    /* coding according to SIII spec */

  /* Service Channel configuration */
  unsigned short  usSvChMDTOffset;      /* coding according to SIII spec */
  unsigned short  usSvChATOffset;       /* coding according to SIII spec */
} SIIIM_DEVICE_OFFSETS_T;

typedef struct SIIIM_CONNECTION_OFFSETS_Ttag
{
  unsigned short  usConnOffset;         /* coding according to SIII spec */
  unsigned short  usConnProcessImageOffset;
  unsigned short  usRtDataProcessImageOffset;
  unsigned short  usRtDataLength;       /* Rt data length excluding conn ctrl size */
  unsigned long   ulConnectionFlags;
} SIIIM_CONNECTION_OFFSETS_T;

/* MDT connection flags */
#define SIIIM_CONNECTION_FLAGS_MDT_INTERNAL_CONNECTION_VALID          0x00000001
#define SIIIM_CONNECTION_FLAGS_MDT_EXCEMPT_FROM_INTERNAL_CCON         0x00000002
#define SIIIM_CONNECTION_FLAGS_MDT_NO_SYNCHRONOUS_TOGGLE              0x00000004

/* AT Connection Handling:
 * Connections to Master need only to be switched on AT to the one with the RT valid information
 * If a switch is not possible within one cycle, the slave is probably dead.
 */

typedef struct SIIIM_CC_CONNECTION_OFFSETS_Ttag
{
  unsigned short  usConnOffset;         /* coding according to SIII spec */
  unsigned short  usRtDataLength;       /* Rt data length excluding conn ctrl size */
} SIIIM_CC_CONNECTION_OFFSETS_T;

/* format of offsets into telegrams */
#define MSK_SIIIM_TELEGRAM_OFFSET_OFFSET        0x07FF

#define MSK_SIIIM_TELEGRAM_OFFSET_TYPE          0x0800
#define VAL_SIIIM_TELEGRAM_OFFSET_TYPE_MDT      0x0800
#define VAL_SIIIM_TELEGRAM_OFFSET_TYPE_AT       0x0000

#define MSK_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER    0xF000    /* only values 0 - 3 valid */
#define SRT_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER    12
#define VAL_SIIIM_TELEGRAM_OFFSET_TEL_NO_0      0x0000
#define VAL_SIIIM_TELEGRAM_OFFSET_TEL_NO_1      0x1000
#define VAL_SIIIM_TELEGRAM_OFFSET_TEL_NO_2      0x2000
#define VAL_SIIIM_TELEGRAM_OFFSET_TEL_NO_3      0x3000

#define SIIIM_MDT_TELEGRAM_OFFSET(telno, offset)                                                            \
          (                                                                                                 \
            VAL_SIIIM_TELEGRAM_OFFSET_TYPE_MDT |                                                            \
            ((telno << SRT_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER) & MSK_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER) |      \
            (offset & MSK_SIIIM_TELEGRAM_OFFSET_OFFSET)                                                     \
          )

#define SIIIM_AT_TELEGRAM_OFFSET(telno, offset)                                                             \
          (                                                                                                 \
            VAL_SIIIM_TELEGRAM_OFFSET_TYPE_AT |                                                             \
            ((telno << SRT_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER) & MSK_SIIIM_TELEGRAM_OFFSET_TEL_NUMBER) |      \
            (offset & MSK_SIIIM_TELEGRAM_OFFSET_OFFSET)                                                     \
          )

/* How to switch CC handling later on, at the moment it will not be implemented
 *
 *
 *   ____             ____         ____
 *  |    |---------->|    |------>|    |-------
 *  | MS |           | S1 |       | S2 |       |
 *  |____|<----------|____|<------|____|<---   |
 *   ^  |                                   |  |
 *   |  |___________________________________|  |
 *   |_________________________________________|
 *
 *
 * CC copying is done so that the CC data is valid on two different lines. We will use the Connection valid for selecting
 * primary/secondary channel.
 *
 * Slaves try to get their data on both directions. Therefore, we only have to bother about a ring separated into two lines.
 */

typedef struct SIIIM_CP3_4_CONFIGURATION_Ttag
{
  /* configuration flags */
  unsigned long                         ulConfigurationFlags;

  /* Process Data Out Size (Master->Slave / MDT) in bytes */
  unsigned long                         ulProcessDataOutSize;
  /* Process Data In Size (Slave->Master / AT) in bytes */
  unsigned long                         ulProcessDataInSize;
  /* MDT sizes */
  unsigned long                         aulMDTSizes[4];         /* size in bytes beginning after HP field */
  /* AT sizes */
  unsigned long                         aulATSizes[4];          /* size in bytes beginning after HP field */

  /* configuration for Device Ctrl / Device Status / Service Channel */
  unsigned long                         ulNumSlaves;
  const SIIIM_DEVICE_OFFSETS_T*         patDeviceOffsets;       /* table has ulNumSlaves entries */

  /* configuration for Process Data Image->MDT connections */
  unsigned long                         ulNumMDTConnections;
  const SIIIM_CONNECTION_OFFSETS_T*     patMDTConnections;      /* table has ulNumConnections entries */

  /* configuration for AT->Process Data Image connections */
  unsigned long                         ulNumATConnections;
  const SIIIM_CONNECTION_OFFSETS_T*     patATConnections;       /* table has ulNumConnections entries */

  /* configuration for CC connections */
  unsigned long                         ulNumCCConnections;
  const SIIIM_CC_CONNECTION_OFFSETS_T*  patCCConnections;       /* table ahs ulNumC2CConnections entries */

  unsigned long                         ulIFGWindowNs;          /* 0 == internal prediction */
  unsigned long                         ulPredictedRingDelayNs; /* for new AT Rx Window handling (0 == internal prediction) */

  struct
  {
    int                                 fEnable;
    unsigned short                      usControlOffset;
    unsigned short                      usStatusOffset;
  } tSCP_SysTime_Control;
} SIIIM_CP3_4_CONFIGURATION_T;

#define MSK_SIIIM_CP3_4_CONFIGURATION_FLAGS_CP3_CP4_EXTERNALLY_TRIGGERED          0x00000001
#define MSK_SIIIM_CP4_ENABLE_EXT_FIELD                                            0x00000002
#define MSK_SIIIM_CP4_ENABLE_INTERNAL_SYNCHRONOUS_CCON                            0x00000004
#define MSK_SIIIM_ENABLE_USE_AT_RECV_TIMER                                        0x00000008
#define MSK_SIIIM_REQUIRE_AT_RECV_TIMER                                           0x00000010

/*
 * Configuration data will be separated and probably recoded in Assembler codes if needed
 * all fields will be internally linked to the particular MDT, AT handlers to reduce the actual work load on
 * given frames.
 *
 * How to deal with missing AT frames on CC?
 *
 * The provided data storage can be released when SIIIM_ConfigureCP3_4 has returned
 */
SIIIM_RESULT
SIIIM_ConfigureCP3_4(
                  SIIIM_HANDLE                        hHandle,
                  const SIIIM_CP3_4_CONFIGURATION_T*  ptCP3_4Config);

SIIIM_RESULT
SIIIM_ClearConfig(
                  SIIIM_HANDLE                        hHandle);

void
SIIIM_SetTsRefMaxCount(
                  SIIIM_HANDLE                        hHandle,
                  unsigned short                      usTsRefMaxCount);

unsigned short
SIIIM_GetTsRefMaxCount(
                  SIIIM_HANDLE                        hHandle);

void
SIIIM_SetActiveRingDelay(
                  SIIIM_HANDLE                        hHandle,
                  unsigned long                       ulActiveRingDelay);

/******************************************************************************
 * Definitions for SIII NRT Channel
 */

#define SIIIM_ETH_TYPE_SERCOSIII      0xCD88        /* word in network byte order on little endian machines */

void
SIIIM_SetupNrtFrameInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void                  (*pfnNrtFrameInt)(SIIIM_HANDLE, unsigned int uPortNo, void*),
                  void*                   pvUserData);

void
SIIIM_DeleteNrtFrameInterrupt(
                  SIIIM_HANDLE            hHandle);

void
SIIIM_GetNrtFrameBufferInfo(
                  SIIIM_HANDLE            hHandle,
                  unsigned long*          pulNumBuffers);

void
SIIIM_GetNrtFrameBuffers(
                  SIIIM_HANDLE            hHandle,
                  void**                  papvFrameHandles);

void
SIIIM_SetPromiscMode(
                  SIIIM_HANDLE            hHandle,
                  unsigned int            fPromiscMode);

unsigned int
SIIIM_NrtSendIsEnabled(
                  SIIIM_HANDLE            hHandle);

/* SIIIM_NrtForwardIsEnabled == SIIIM_EnableNrtForward(fEnable) && InternalForwardEnableConditions */
unsigned int
SIIIM_NrtForwardIsEnabled(
                  SIIIM_HANDLE            hHandle);

void
SIIIM_EnableNrtForward(
                  SIIIM_HANDLE             hHandle,
                  unsigned int             fEnable);

void
SIIIM_EnableNrtForwardInNRT(
                  SIIIM_HANDLE             hHandle);

void
SIIIM_ResetNrtForwardInNRT(
                  SIIIM_HANDLE             hHandle);

unsigned int
SIIIM_SendNrtRingCheckFramesEnabled(
                  SIIIM_HANDLE             hHandle);

void
SIIIM_DecreaseNrtRingCheckCounter(
                  SIIIM_HANDLE             hHandle);

/******************************************************************************
 * Definitions for NRT Link Status Handling
 */

SIIIM_RESULT
SIIIM_SetupNrtLinkStatusInterrupt(
                  SIIIM_HANDLE            hHandle,
                  void                  (*pfnNrtLinkStatusInt)( SIIIM_HANDLE,
                                                                unsigned int uiPortNo,
                                                                void*),
                  void*                   pvUserData);

void
SIIIM_DeleteNrtLinkStatusInterrupt(
                  SIIIM_HANDLE            hHandle);

/******************************************************************************
 * Conversion function for SIIIM_NRT
 */
void*
SIIIM_ConvertFramePtrFromFifoElem(
    SIIIM_HANDLE hHandle,
    void* hFrame);

/******************************************************************************
 * Diag Counters
 */
typedef struct SIIIM_COUNTERS_Ttag
{
  unsigned long ulFramesTransmittedOk;               /**< Number of successfully transmitted frames */
  unsigned long ulSingleCollisionFrames;             /**< Number of single collisions during transmission */
  unsigned long ulMultipleCollisionFrames;           /**< Number of multiple collisions during transmission */
  unsigned long ulLateCollisions;                    /**< Number of late collisions during transmission */
  unsigned long ulLinkDownDuringTransmission;        /**< Number of transmission request while
                                                          Ethernet link was down */
  unsigned long ulUtxUnderflowDuringTransmission;    /**< Number of internal FIFO underflows during transmission */
  unsigned long ulTxDroppedDueNrtWindowTooSmall;     /**< Number of frames dropped because NRT windows was
                                                          smaller than frame length */
  unsigned long ulTxFatalErrors;                     /**< Number of fatal transmission errors */
  unsigned long ulFramesReceivedOk;                  /**< Number of correctly received frames */
  unsigned long ulFrameCheckSequenceErrors;          /**< Number of received frames with checksum error */
  unsigned long ulAlignmentErrors;                   /**< Number of received frames with alignment error */
  unsigned long ulFrameTooLongErrors;                /**< Number of received frames with length error */
  unsigned long ulRuntFramesReceived;                /**< Number of received runt frames */
  unsigned long ulCollisionFragmentsReceived;        /**< Number of received collision fragments */
  unsigned long ulFramesDroppedDueLowResource;       /**< Number of received frames dropped because no
                                                          more free frame buffers are available */
  unsigned long ulFramesDroppedDueUrxOverflow;       /**< Number of received frames dropped due to
                                                          internal FIFO overflow */
  unsigned long ulRxFatalErrors;                     /**< Number of fatal receive errors */
  unsigned long ulMdtNotUpdatedErrors;               /**< Number of MDT not updated errors */
  unsigned long ulAtNotUpdatedErrors;                /**< Number of AT not updated errors */
} SIIIM_COUNTERS_T;

void
SIIIM_GetCounters(SIIIM_HANDLE hHandle, unsigned int uiPortNo, SIIIM_COUNTERS_T* ptCounters);

/******************************************************************************
 * Internal Test Functions
 */
typedef enum SIIIM_TELEGRAM_ID_Ttag
{
  SIIIM_TELEGRAM_MDT0,//!< SIIIM_TELEGRAM_MDT0
  SIIIM_TELEGRAM_MDT1,//!< SIIIM_TELEGRAM_MDT1
  SIIIM_TELEGRAM_MDT2,//!< SIIIM_TELEGRAM_MDT2
  SIIIM_TELEGRAM_MDT3,//!< SIIIM_TELEGRAM_MDT3
  SIIIM_TELEGRAM_AT0, //!< SIIIM_TELEGRAM_AT0
  SIIIM_TELEGRAM_AT1, //!< SIIIM_TELEGRAM_AT1
  SIIIM_TELEGRAM_AT2, //!< SIIIM_TELEGRAM_AT2
  SIIIM_TELEGRAM_AT3  //!< SIIIM_TELEGRAM_AT3
} SIIIM_TELEGRAM_ID_T;

void
SIIIM_SetTelegramType(
                  SIIIM_HANDLE            hHandle,
                  SIIIM_TELEGRAM_ID_T     eTelegramId,
                  unsigned char           bTelegramType);

unsigned char
SIIIM_GetTelegramType(
                  SIIIM_HANDLE            hHandle,
                  SIIIM_TELEGRAM_ID_T     eTelegramId);

signed long
SIIIM_GetMstSendTimestampDifference(
                  SIIIM_HANDLE            hHandle);

long
SIIIM_GetSysTimeAddUpCorrection(
    SIIIM_HANDLE hHandle);

unsigned long
SIIIM_GetSystemJitterMargin(
    SIIIM_HANDLE hHandle);

/******************************************************************************
 * Perf counter (only on specific implementations)
 */
unsigned long
SIIIM_GetIdleCounter(SIIIM_HANDLE           hHandle);

int
SIIIM_PerfCounterValid(SIIIM_HANDLE         hHandle);

#endif // SIIIM_HAL_H_INCLUDED
